<?php
/**
 * Transfer Character
 * https://webenginecms.org/
 * 
 * @version 1.1.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2021 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\TransferCharacter;

class TransferCharacter {
	
	private $_configXml = 'config.xml';
	private $_modulesPath = 'modules';
	private $_serverFiles;
	private $_sqlTable = WE_PREFIX . 'WEBENGINE_TRANSFER_CHARACTER_LOGS';
	private $_sqlPath = 'sql';
	
	private $_cost;
	private $_configId;
	private $_requiredLevel;
	private $_requiredMasterLevel;
	private $_requiredResets;
	
	private $_userid;
	private $_username;
	private $_character;
	private $_newUserid;
	
	private $_usercpmenu = array(
		array(
			'active' => true,
			'type' => 'internal',
			'phrase' => 'transfercharacter_title',
			'link' => 'usercp/transfercharacter',
			'icon' => 'usercp_default.png',
			'visibility' => 'user',
			'newtab' => false,
			'order' => 999,
		),
	);
	
	// CONSTRUCTOR
	
	function __construct() {
		
		// webengine configs
		$this->config = webengineConfigs();
		$this->_serverFiles = strtolower($this->config['server_files']);
		
		// load databases
		$this->common = new \common();
		$this->mu = \Connection::Database('MuOnline');
		$this->me = \Connection::Database('Me_MuOnline');
		
		// load configs
		$this->configFilePath = __PATH_TRANSFERCHARACTER_ROOT__.$this->_configXml;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('transfercharacter_error_2'));
		$xml = simplexml_load_file($this->configFilePath);
		if(!$xml) throw new \Exception(lang('transfercharacter_error_2'));
		
		// set configs
		$this->_cost = $xml->transfer_cost;
		$this->_configId = $xml->config_id;
		$this->_requiredLevel = $xml->level_required;
		$this->_requiredMasterLevel = $xml->masterlevel_required;
		$this->_requiredResets = $xml->resets_required;
		
		// sql file path
		$this->sqlFilePath = __PATH_TRANSFERCHARACTER_ROOT__.$this->_sqlPath.'/';
		
		// check tables
		$this->_checkTable();
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('transfercharacter_error_3'));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('transfercharacter_error_3'));
		if(!@include_once(__PATH_TRANSFERCHARACTER_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('transfercharacter_error_3'));
	}
	
	public function setUserid($userid) {
		$this->_userid = $userid;
	}
	
	public function setUsername($username) {
		if(!\Validator::UsernameLength($username)) throw new \Exception(lang('transfercharacter_error_10'));
		$this->_username = $username;
	}
	
	public function setCharacter($character) {
		$this->_character = $character;
	}
	
	public function setTransferId($id) {
		if(!\Validator::UnsignedNumber($id)) throw new \Exception(lang('transfercharacter_error_16'));
		$result = $this->decodeTransferId($id);
		if(!\Validator::UnsignedNumber($result)) throw new \Exception(lang('transfercharacter_error_16'));
		$this->_newUserid = $result;
	}
	
	public function getCost() {
		return $this->_cost;
	}
	
	public function getRequiredLevel() {
		return $this->_requiredLevel;
	}
	
	public function getRequiredMasterLevel() {
		return $this->_requiredMasterLevel;
	}
	
	public function getRequiredResets() {
		return $this->_requiredResets;
	}
	
	public function getAccountCharacterList() {
		if(!check_value($this->_username)) throw new \Exception(lang('transfercharacter_error_4'));
		$Character = new \Character();
		$AccountCharacters = $Character->AccountCharacter($this->_username);
		if(!is_array($AccountCharacters)) throw new \Exception(lang('transfercharacter_error_5'));
		return $AccountCharacters;
	}
	
	public function transfer() {
		if($this->_configId == 0) throw new \Exception(lang('transfercharacter_error_15'));
		if(!check_value($this->_userid)) throw new \Exception(lang('transfercharacter_error_7'));
		if(!check_value($this->_username)) throw new \Exception(lang('transfercharacter_error_7'));
		if(!check_value($this->_character)) throw new \Exception(lang('transfercharacter_error_7'));
		if(!check_value($this->_newUserid)) throw new \Exception(lang('transfercharacter_error_7'));
		
		// check if transfering to same account
		if($this->_newUserid == $this->_userid) throw new \Exception(lang('transfercharacter_error_17'));
		
		// account object
		$Account = new \Account();
		
		// account info
		$accountInfo = $Account->accountInformation($this->_userid);
		if(!is_array($accountInfo)) throw new \Exception(lang('transfercharacter_error_11'));
		if($Account->accountOnline($this->_username)) throw new \Exception(lang('transfercharacter_error_12'));
		
		// current owner character slot
		$currentOwnerCharacterSlot = $this->_getCurrentOwnerCharacterSlot();
		
		// new owner account info
		$newOwnerAccountInfo = $Account->accountInformation($this->_newUserid);
		if(!is_array($newOwnerAccountInfo)) throw new \Exception(lang('transfercharacter_error_18'));
		if($Account->accountOnline($newOwnerAccountInfo[_CLMN_USERNM_])) throw new \Exception(lang('transfercharacter_error_19'));
		
		// empty slot in new owners accountcharacter
		$newOwnerEmptySlot = $this->_getAccountCharacterEmptySlot($newOwnerAccountInfo[_CLMN_USERNM_]);
		if(!$newOwnerEmptySlot) throw new \Exception(lang('transfercharacter_error_14'));
		
		// check credits
		$creditSystem = new \CreditSystem();
		$creditSystem->setConfigId($this->_configId);
		$configInfo = $creditSystem->showConfigs(true);
		switch($configInfo['config_user_col_id']) {
			case 'userid':
				$creditSystem->setIdentifier($accountInfo[_CLMN_MEMBID_]);
				break;
			case 'username':
				$creditSystem->setIdentifier($accountInfo[_CLMN_USERNM_]);
				break;
			case 'email':
				$creditSystem->setIdentifier($accountInfo[_CLMN_EMAIL_]);
				break;
			case 'character':
				$creditSystem->setIdentifier($this->_character);
				break;
			default:
				throw new \Exception(lang('transfercharacter_error_11'));
		}
		
		// check cost
		if($this->_cost > $creditSystem->getCredits()) throw new \Exception(lang('transfercharacter_error_13'));
		
		// check character data
		$Character = new \Character();
		$characterData = $Character->CharacterData($this->_character);
		
		// check level
		if($this->_requiredLevel > 0) {
			if($characterData[_CLMN_CHR_LVL_] < $this->_requiredLevel) throw new \Exception(lang('transfercharacter_error_8'));
		}
		
		// check resets
		if($this->_requiredResets > 0) {
			if($characterData[_CLMN_CHR_RSTS_] < $this->_requiredResets) throw new \Exception(lang('transfercharacter_error_24'));
		}
		
		// check master level
		if($this->_requiredMasterLevel > 0) {
			$masterLevelData = $Character->getMasterLevelInfo($this->_character);
			if($characterData[_CLMN_ML_LVL_] < $this->_requiredMasterLevel) throw new \Exception(lang('transfercharacter_error_25'));
		}
		
		// subtract credits
		$creditSystem->subtractCredits($this->_cost);
		
		// Transfer Process
		$transfer1 = $this->_transferProcess_removeFromCurrentOwnerAccountCharacter($currentOwnerCharacterSlot);
		if(!$transfer1) throw new \Exception(lang('transfercharacter_error_20'));
		
		$transfer2 = $this->_transferProcess_insertToNewOwnerAccountCharacter($newOwnerEmptySlot, $newOwnerAccountInfo[_CLMN_USERNM_]);
		if(!$transfer2) throw new \Exception(lang('transfercharacter_error_21'));
		
		$transfer3 = $this->_transferProcess_updateCharacterAccount($newOwnerAccountInfo[_CLMN_USERNM_]);
		if(!$transfer3) throw new \Exception(lang('transfercharacter_error_22'));
		
		// save log
		$this->_addTransferLog($accountInfo[_CLMN_USERNM_], $newOwnerAccountInfo[_CLMN_USERNM_], $this->_character);
		
		// redirect
		redirect(1, 'usercp/transfercharacter/success/1');
	}
	
	public function generateTransferId($userid) {
		return (((($userid*2020)/2)-500)/2)*83;
	}
	
	public function decodeTransferId($id) {
		return (((($id/83)*2)+500)*2)/2020;
	}
	
	public function getTransferLogs($limit=50) {
		if($limit == 0) {
			$result = $this->me->query_fetch("SELECT * FROM ".$this->_sqlTable." ORDER BY id DESC");
		} else {
			$result = $this->me->query_fetch("SELECT TOP ".$limit." * FROM ".$this->_sqlTable." ORDER BY id DESC");
		}
		if(!is_array($result)) return;
		return $result;
	}
	
	public function checkPluginUsercpLinks() {
		if(!is_array($this->_usercpmenu)) return;
		$cfg = loadConfig('usercp');
		if(!is_array($cfg)) return;
		foreach($cfg as $usercpMenu) {
			$usercpLinks[] = $usercpMenu['link'];
		}
		foreach($this->_usercpmenu as $pluginMenuLink) {
			if(in_array($pluginMenuLink['link'],$usercpLinks)) continue;
			$cfg[] = $pluginMenuLink;
		}
		usort($cfg, function($a, $b) {
			return $a['order'] - $b['order'];
		});
		$usercpJson = json_encode($cfg, JSON_PRETTY_PRINT);
		$cfgFile = fopen(__PATH_CONFIGS__.'usercp.json', 'w');
		if(!$cfgFile) throw new \Exception('There was a problem opening the usercp file.');
		fwrite($cfgFile, $usercpJson);
		fclose($cfgFile);
	}
	
	// PROTECTED FUNCTIONS
	
	protected function _getAccountCharacterEmptySlot($username) {
		$accountCharacter = $this->mu->query_fetch_single("SELECT * FROM "._TBL_AC_." WHERE "._CLMN_AC_ID_." = ?", array($username));
		if(!is_array($accountCharacter)) return;
		if(array_key_exists('GameID1', $accountCharacter)) if(!check_value($accountCharacter['GameID1'])) return 'GameID1';
		if(array_key_exists('GameID2', $accountCharacter)) if(!check_value($accountCharacter['GameID2'])) return 'GameID2';
		if(array_key_exists('GameID3', $accountCharacter)) if(!check_value($accountCharacter['GameID3'])) return 'GameID3';
		if(array_key_exists('GameID4', $accountCharacter)) if(!check_value($accountCharacter['GameID4'])) return 'GameID4';
		if(array_key_exists('GameID5', $accountCharacter)) if(!check_value($accountCharacter['GameID5'])) return 'GameID5';
		if(array_key_exists('GameID6', $accountCharacter)) if(!check_value($accountCharacter['GameID6'])) return 'GameID6';
		if(array_key_exists('GameID7', $accountCharacter)) if(!check_value($accountCharacter['GameID7'])) return 'GameID7';
		if(array_key_exists('GameID8', $accountCharacter)) if(!check_value($accountCharacter['GameID8'])) return 'GameID8';
		return;
	}
	
	protected function _getCurrentOwnerCharacterSlot() {
		if(!check_value($this->_username)) return;
		if(!check_value($this->_character)) return;
		$accountCharacter = $this->mu->query_fetch_single("SELECT * FROM "._TBL_AC_." WHERE "._CLMN_AC_ID_." = ?", array($this->_username));
		if(!is_array($accountCharacter)) return;
		if(array_key_exists('GameID1', $accountCharacter)) if($accountCharacter['GameID1'] == $this->_character) return 'GameID1';
		if(array_key_exists('GameID2', $accountCharacter)) if($accountCharacter['GameID2'] == $this->_character) return 'GameID2';
		if(array_key_exists('GameID3', $accountCharacter)) if($accountCharacter['GameID3'] == $this->_character) return 'GameID3';
		if(array_key_exists('GameID4', $accountCharacter)) if($accountCharacter['GameID4'] == $this->_character) return 'GameID4';
		if(array_key_exists('GameID5', $accountCharacter)) if($accountCharacter['GameID5'] == $this->_character) return 'GameID5';
		if(array_key_exists('GameID6', $accountCharacter)) if($accountCharacter['GameID6'] == $this->_character) return 'GameID6';
		if(array_key_exists('GameID7', $accountCharacter)) if($accountCharacter['GameID7'] == $this->_character) return 'GameID7';
		if(array_key_exists('GameID8', $accountCharacter)) if($accountCharacter['GameID8'] == $this->_character) return 'GameID8';
		return;
	}
	
	protected function _transferProcess_removeFromCurrentOwnerAccountCharacter($slot) {
		$accountCharacter = $this->mu->query_fetch_single("SELECT * FROM "._TBL_AC_." WHERE "._CLMN_AC_ID_." = ?", array($this->_username));
		if(!is_array($accountCharacter)) return;
		$result = $this->mu->query("UPDATE "._TBL_AC_." SET ".$slot." = NULL WHERE "._CLMN_AC_ID_." = ?", array($this->_username));
		if(!$result) return;
		return true;
	}
	
	protected function _transferProcess_insertToNewOwnerAccountCharacter($slot, $username) {
		if(!check_value($this->_character)) return;
		$result = $this->mu->query("UPDATE "._TBL_AC_." SET ".$slot." = ? WHERE "._CLMN_AC_ID_." = ?", array($this->_character, $username));
		if(!$result) return;
		return true;
	}
	
	protected function _transferProcess_updateCharacterAccount($newUsername) {
		if(!check_value($this->_character)) return;
		$result = $this->mu->query("UPDATE "._TBL_CHR_." SET "._CLMN_CHR_ACCID_." = ? WHERE "._CLMN_CHR_NAME_." = ?", array($newUsername, $this->_character));
		if(!$result) return;
		return true;
	}
	
	protected function _addTransferLog($oldAccount, $newAccount, $characterName) {
		$result = $this->me->query("INSERT INTO ".$this->_sqlTable." (original_owner_username, new_owner_username, character_name, transfer_date) VALUES (?, ?, ?, CURRENT_TIMESTAMP)", array($oldAccount, $newAccount, $characterName));
		if(!$result) return;
		return true;
	}
	
	// PRIVATE FUNCTIONS
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_TRANSFERCHARACTER_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _checkTable() {
		$tableExists = $this->me->query_fetch_single("SELECT * FROM sysobjects WHERE xtype = 'U' AND name = ?", array($this->_sqlTable));
		if($tableExists) return true;
		if(!$this->_createTable()) throw new \Exception(lang('transfercharacter_error_23', true));
	}
	
	private function _createTable() {
		if(!file_exists($this->sqlFilePath.'WEBENGINE_TRANSFER_CHARACTER_LOGS.txt')) return;
		$query = file_get_contents($this->sqlFilePath.'WEBENGINE_TRANSFER_CHARACTER_LOGS.txt');
		if(!check_value($query)) return;
		$queryFinal = str_replace('{TABLE_NAME}', $this->_sqlTable, $query);
		if(!$queryFinal) return;
		if(!$this->me->query($queryFinal)) return;
		return true;
	}
}